#include "WordFilterSony.h"
#include "MessagePipe.h"
#include "ErrorCodesSony.h"
#include "SignInSony.h"

using namespace sce::Toolkit::NP;
using namespace sce::Toolkit::NP::Utilities;

namespace UnityPlugin
{
	CachedWordFilter gWordFilter;

	DO_EXPORT( bool, PrxWordFilterIsBusy ) ()
	{
		return gWordFilter.IsBusy();
	}

	DO_EXPORT( bool, PrxWordFilterGetLastError ) (ResultCode* result)
	{
		return gWordFilter.GetLastError(result);
	}

	DO_EXPORT( ErrorCode, PrxWordFilterCensorComment ) (const char* comment)
	{
		return gWordFilter.CensorComment(comment);
	}

	DO_EXPORT( ErrorCode, PrxWordFilterSanitizeComment ) (const char* comment)
	{
		return gWordFilter.SanitizeComment(comment);
	}

	DO_EXPORT( ErrorCode, PrxWordFilterGetResult ) (FilteredComment* result)
	{
		return gWordFilter.GetResult(result);
	}

	bool CachedWordFilter::IsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_Busy;
	}

	ErrorCode CachedWordFilter::GetResult(FilteredComment* result)
	{
		result->wasChanged = m_CommentWasChanged;
		result->comment = m_FilteredComment.c_str();
		return NP_OK;
	}

	CachedWordFilter::CachedWordFilter()
		: m_Busy(false)
		, m_LastResult("WordFilter")
		, m_CommentWasChanged(false)
		, m_WordFilterParam(NULL)
	{
	}

	CachedWordFilter::~CachedWordFilter()
	{
		delete m_WordFilterParam;
	}

	bool CachedWordFilter::ProcessEvent(const sce::Toolkit::NP::Event& event)
	{
		SimpleLock::AutoLock lock(m_Lock);
		bool handled = false;

		switch(event.event)
		{
		case Event::wordFilterComplete:			// An event generated when a word filter operation has been completed.	(this is what you always get on PS3 - initSuccess and Failed don't exist)
		//	DELIBERATE FALL THROUGH
			if(m_FutureSanitized.hasResult())
			{
				m_CommentWasChanged = m_FilteredComment.compare(m_FutureSanitized.get()->sanitizedComment) != 0;
				//	On PS3, sanitized strings can come back with trailing characters on the end of the result. Since the result cannot exceed the length of the original string
				//	we can truncate the result here to be the same length as the original.
				int	lenOriginal = strlen( m_WordFilterParam->comment );
				m_FutureSanitized.get()->sanitizedComment[lenOriginal] = 0;
				m_FilteredComment = m_FutureSanitized.get()->sanitizedComment;
				m_FutureSanitized.reset();
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_WordFilterSanitized);
			}
			else
			{
				switch(event.returnCode)
				{
				case SCE_TOOLKIT_NP_SUCCESS:
					UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_WordFilterNotCensored);
					break;

				case SCE_NP_COMMUNITY_SERVER_ERROR_CENSORED:
					m_FilteredComment = "";
					m_CommentWasChanged = true;
					UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_WordFilterCensored);
					break;

				default:
					m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_WordFilterError);
					break;
				}
			}
			m_Busy = false;
			handled = true;
			break;

		default:
			UnityPlugin::Messages::LogWarning("Unexpected event from word filter service: event=%d\n", event.event);
			handled = true;
			break;
		}

		return handled;
	}

	ErrorCode CachedWordFilter::SanitizeComment(const char* comment)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();
		m_FilteredComment = comment;
		m_CommentWasChanged = false;

		if(m_WordFilterParam)
		{
			delete m_WordFilterParam;
		}

		m_WordFilterParam = new WORDFILTERPARAM();
		memset(m_WordFilterParam, 0, sizeof(WORDFILTERPARAM));

		strncpy(m_WordFilterParam->comment, comment, SCE_NP_SCORE_CENSOR_COMMENT_MAXLEN);
		m_WordFilterParam->comment[SCE_NP_SCORE_CENSOR_COMMENT_MAXLEN-1] = 0;	//	Force truncated termination


		m_WordFilterParam->isAsync = true;
		int ret = sce::Toolkit::NP::WordFilter::Interface::filterWord(&m_FutureSanitized, m_WordFilterParam);

		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedWordFilter::CensorComment(const char* comment)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();
		m_FilteredComment = comment;
		m_CommentWasChanged = false;

		if(m_WordFilterParam)
		{
			delete m_WordFilterParam;
		}
		m_WordFilterParam = new WORDFILTERPARAM();
		memset(m_WordFilterParam, 0, sizeof(WORDFILTERPARAM));
		strncpy(m_WordFilterParam->comment, comment, SCE_NP_SCORE_CENSOR_COMMENT_MAXLEN);
		m_WordFilterParam->comment[SCE_NP_SCORE_CENSOR_COMMENT_MAXLEN-1] = 0;	//	Force truncated termination


		m_WordFilterParam->isAsync = true;
		int ret = sce::Toolkit::NP::WordFilter::Interface::filterWord(NULL, m_WordFilterParam);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;
		return m_LastResult.GetResult();
	}
}
